function [Edge, Etri, Boundary, Vmem, Vnodal, Enodal, Nodal, Nodal_Bud, Nodes, Ntri, Nmem] = Lists( tri, vert )
%Given a triangulation and vertex list output Edge, Etri and Boundary lists
%needed for FEM.  An important trick: Tests for equality are only reliable
%if applied to integers!  Otherwise numerical error won't recognize
%equality between things which should be equal.  It's for this reason that
%I wrote the Nodal and Nodal_Bud lists.  They needed to be given as indices
%of the vertex and edge lists so that equality -- when came to define the
%nodal basis functions -- was exact!
%   Edge is a numedge x 4 list.  The first two are the vertex indices of
%   the edge and the second two record to which triangles the edge belongs.
%   0 is vacuous. Boundary is a numbdtri x 4 array.  First entry is the
%   index of the boundary triangle.  The second is the index of its
%   corresponding boundary edge.  The last two are the indices of the
%   boundary vertices on that edge.  Etri is a numtri x 3 array which gives
%   triangles by their edges as indexed by Edge. Boundary is a numbdtri x 4
%   array.  The first column is index of boundary triangle (index by tri).
%   The second is index of boundary edge (index by Edge). The last two
%   columns are indices of boundary vertices for that edge (index by vert).
%   Vmem is a numvert x numtri array.  Each row lists the indices of the
%   triangle to which vert(row,:) belongs. VNodal gives the index of
%   vertices (by vert) but organized so that first first numbdtri entries
%   are the boundary vertices. ENodal is similar but indexed like Edge and
%   the first numbdtri are the boundary edges. Nodal is a combination index
%   list.  The first 1:Nodal_Bud(1,1) entries are indices of interior
%   vertices, indexed by vert.  The next Nodal_Bud(1,1)+1:Nodal_Bud(2,1)
%   entries are indices of interior edges (whose midpoints are nodes!),
%   indexed by Edge. Then the Nodal_Bud(2,1)+1:Nodal_Bud(3,1) are boundary
%   vertices indexed by vert.  Finally, Nodal_Bud(3,1)+1:Nodal_Bud(4,1) are
%   the indices of boundary edges. Nodal_Bud is a 4 x 1 list that just
%   tells me how Nodal is divided.  Conveniently all the boundary nodes
%   correspond to entries Nodal_Bud(2,1)+1:Nodal_Bud(4,1).  Each entry of
%   Bud is the last index of Nodal corresponding to a certain type of
%   int/bd vert/edge. Nodes gives the x-y locations of the Nodes referred
%   to by Nodal, either by vert x-y coordinates or the midpoint of given
%   edge. Ntri is a numtri x 6 array that gives triangles by their nodes as
%   indexed by nodal.  First three are vertices, second three are
%   edges(=midpoints). Finally, Nmem, like Vmem lists the triangles --
%   indexed like tri -- to which a given Node indexed like Nodal/Nodes
%   (equivalent!) belong.


numtri = size(tri,1);
numvert = size(vert,1);

%%%%%% EDGE LIST  and Triangles by Edge Lists%%%%%%
%We first setup an edge list
Edge = zeros(3*numtri, 4); %Maximum Number of edges if tri's disjoint. 
                           %First 
                           %two columns record v1,v2 indexed by vert. 
                           %The third records to which was the first
                           %triangle that the edge belonged. The fourth if 0
                           %means the edge only belonged to a single
                           %triangle, hence is boundary.  If not 0, it
                           %records the second triangle to which it
                           %belonged. No edge is shared by more than two
                           %triangles.
                           
%We also setup a triangle by edges list
Etri = zeros(numtri, 3);
                            %This list will write out triangles but given
                            %according to the index of their three edges,
                            %not vertices.  Note: Etri(i) refers to tri(i),
                            %so the cycle over triangles is the same.
                            
%First triangle
Edge(1:3,:) = [[tri(1,1) tri(1,2) 1 0];[tri(1,2) tri(1,3) 1 0]; [tri(1,3) tri(1,1) 1 0]];
Etri(1,1:3) = [[1 2 3]];


%Now we cycle through all other triangles, adding what's new and recording
%what's doubled.

count = 3; %Count is # of distinct edges
for i=2:numtri
    
    %Careful of edge orientation!
    
    %Check to see if first edge of triangle has been recorded. 
    j = 1;
     while ( (tri(i,1) ~= Edge(j,1)) || (tri(i,2) ~= Edge(j,2)) ) && ( (tri(i,2) ~= Edge(j,1)) || (tri(i,1) ~= Edge(j,2)) ) && j<= count %Make sure in some component each ordering isn't equal.
        j = j + 1; %Cycle to next edge
    end
    if j == count + 1 %New Edge so record it with frequency.
        count = count + 1; %Update number of edges
        Edge(count,:) = [[tri(i,1) tri(i,2) i 0]];
        Etri(i,1) = count;
    else %Repeat edge so record to which triangle belongs.
        Edge(j,4) = i;
        Etri(i,1) = j;
    end
    
    
    %Check to see if 2nd edge of triangle has been recorded
    j = 1;
      while ( (tri(i,2) ~= Edge(j,1)) || (tri(i,3) ~= Edge(j,2)) ) && ( (tri(i,3) ~= Edge(j,1)) || (tri(i,2) ~= Edge(j,2)) ) && j<= count %Make sure in some component each ordering isn't equal.
        j = j + 1; %Cycle to next edge
    end
    if j == count + 1 %New Edge so record it with frequency.
        count = count + 1; %Update number of edges
        Edge(count,:) = [[tri(i,2) tri(i,3) i 0]];
        Etri(i,2) = count;
    else %Repeat edge so record to which triangle belongs.
        Edge(j,4) = i;
        Etri(i,2) = j;
    end
   
    
    %Check to see if 3rd edge of triangle has been recorded
    j = 1;
     while ( (tri(i,3) ~= Edge(j,1)) || (tri(i,1) ~= Edge(j,2)) ) && ( (tri(i,1) ~= Edge(j,1)) || (tri(i,3) ~= Edge(j,2)) ) && j<= count %Make sure in some component each ordering isn't equal.
        j = j + 1; %Cycle to next edge
    end
    if j == count + 1 %New Edge so record it with frequency.
        count = count + 1; %Update number of edges
        Edge(count,:) = [[tri(i,3) tri(i,1) i 0]];
        Etri(i,3) = count;
    else %Repeat edge so record to which triangle belongs.
        Edge(j,4) = i;
        Etri(i,3) = j;
    end
    
end

Edge(count+1:3*numtri,:) = []; %Delete the padding which didn't need
numedge = size(Edge,1); %Record number of edges


%%%%%% Boundary List %%%%%%%
%Boundary is numbdtri x 4.  The first column is the triangle, 2nd is it's
%boundary edge, and the last two are the two boundary vertices.

Boundary = zeros(3*numtri,4); %Again build for max it could be

count = 0; %Count number of boundary edges.
for i=1:numedge
    if Edge(i,4) == 0 %Is boundary Edge
        count = count + 1; %Update count
        Boundary(count,:) = [[Edge(i,3) i Edge(i,1:2)]];
    end
end

Boundary(count+1:3*numtri,:) = []; %Delete the excess
clear count 
%%%%%%%% VMem %%%%%%%%%%%
%List those triangles to which a given vertex belongs.
Vmem = zeros(size(vert,1),numtri); %Maximum number of triangles a vertex can belong to

for i=1:numtri
    %In the ith triangle, fill in the nodal indices in the space available
    for j=1:3
        k=1;
        while Vmem(tri(i,j),k) ~= 0 && k < numtri
            k = k + 1;
        end
    Vmem(tri(i,j),k) = i;
    end
end

%Cut out the unnecessary zeros
S = sum(Vmem);
k=1;
while k<= size(Vmem,2) && S(1,k)~=0
    k = k + 1;
end
Vmem(:,k:numtri) = [];

%%%%%%% Nodal Lists %%%%%%%%%
%The Nodes are vertices and midpoints of edges.  We want to keep track of
%where the boundary nodes are, so we write  a sort to handle that. Note:
%The first size(Boundary,1) entries of VNodal and ENodal are the boundary
%vertices/edges respectively. The rest are interior.
Vnodal = zeros(size(vert,1)+2*size(Boundary,1),1); 
Enodal = zeros(size(Edge,1)+size(Boundary,1),1);

ABC = zeros(size(vert,1),1);%This just indexes vertices according to how listed in vert.  ie [1 2 3 4 ...]'
for i=1:size(vert,1)
    ABC(i,1) = i;
end
Vnodal = [Boundary(:,3); Boundary(:,4); ABC]; %Put boundary vertices first. This creates a redundant list.

%Run a backwards parse through VNodal to eliminate redundancy. You go
%backwards so that the indexing doesn't change as you delete entries.
count = size(Vnodal,1);
for i=1:count-1 %Because the first entry can't be a duplicate of any preceeding
    k=1;
    while count + 1 - i - k >= 1 && Vnodal(count+1-i,1) ~= Vnodal(count+1-i-k,1)
        k = k + 1;
    end
    if k ~= count + 1 - i %Means a redundant edge
        Vnodal(count+1-i,:)=[];
    end
end


ABC = zeros(size(Edge,1),1);
for i=1:size(Edge,1)
    ABC(i,1) = i;
end
 
Enodal = [Boundary(:,2); ABC];  %Put boundary edges first.  Again this creates a redundant list.
count = size(Enodal,1);
%Again backwards redundancy check on ENodal
for i=1:count-1 %Because the first entry can't be a duplicate of any preceeding
    k=1;
    while count + 1 - i - k >= 1 && Enodal(count+1-i,1) ~= Enodal(count+1-i-k,1)
        k = k + 1;
    end
    if k ~= count + 1 - i %Means a redundant edge
        Enodal(count+1-i,:)=[];
    end
end

%The indices of nodal intermix vertex indices and edge indices
Nodal = zeros(size(Vnodal,1)+size(Enodal,1),1);
Nodal_Bud = zeros(4,1); %Lists the last index of nodal which corresponds to that type, eg. int vertex, int edge, bounday vertex, boundary edge

Nodal_Bud(1,1) = size(Vnodal,1)-size(Boundary,1);
Nodal_Bud(2,1) = size(Vnodal,1)-size(Boundary,1) + size(Enodal,1)-size(Boundary,1);
Nodal_Bud(3,1) = size(Vnodal,1)-size(Boundary,1) + size(Enodal,1);
Nodal_Bud(4,1) = size(Vnodal,1) + size(Enodal,1);

Nodal(1:size(Vnodal,1)-size(Boundary,1),1) = Vnodal(size(Boundary,1)+1:size(Vnodal,1),1); %Copy in INTERIOR VERTICES
Nodal(size(Vnodal,1)-size(Boundary,1)+1:size(Vnodal,1)-size(Boundary,1) + size(Enodal,1)-size(Boundary,1),1) = Enodal(size(Boundary,1)+1:size(Enodal,1),1); %Copy in INTERIOR EDGES
Nodal(size(Vnodal,1)-size(Boundary,1) + size(Enodal,1)-size(Boundary,1)+1:size(Vnodal,1)-size(Boundary,1) + size(Enodal,1),1) = Vnodal(1:size(Boundary,1),1); %Copy in BOUNDARY VERTICES
Nodal(size(Vnodal,1)-size(Boundary,1) + size(Enodal,1)+1:size(Vnodal,1) + size(Enodal,1),1) = Enodal(1:size(Boundary,1),1); %Copy in BOUNDARY EDGES

%Here we record the locations of the Nodes
Nodes = zeros(size(Nodal,1),2);

Nodes(1:size(Vnodal,1)-size(Boundary,1),:) = vert(Vnodal(size(Boundary,1)+1:size(Vnodal,1),1),:); %Copy in location INTERIOR VERTICES
Nodes(size(Vnodal,1)-size(Boundary,1)+1:size(Vnodal,1)-size(Boundary,1) + size(Enodal,1)-size(Boundary,1),:) = 1/2*vert(Edge(Enodal(size(Boundary,1)+1:size(Enodal,1),1) ,1) ,:) + 1/2*vert(Edge(Enodal(size(Boundary,1)+1:size(Enodal,1),1) ,2), :);%Copy in location INTERIOR EDGE midpoints
Nodes(size(Vnodal,1)-size(Boundary,1) + size(Enodal,1)-size(Boundary,1)+1:size(Vnodal,1)-size(Boundary,1) + size(Enodal,1),:) = vert(Vnodal(1:size(Boundary,1),1),:); %Copy in location BOUNDARY VERTICES
Nodes(size(Vnodal,1)-size(Boundary,1) + size(Enodal,1)+1:size(Vnodal,1) + size(Enodal,1),:) = 1/2*vert( Edge( Enodal(1:size(Boundary,1),1),1), :) + 1/2*vert( Edge( Enodal(1:size(Boundary,1),1),2 ), :); %Copy in location BOUNDARY EDGES midpoints

%%%%%%% NTri %%%%%%%%
%Here we index triangles according to the index of their nodal members.
%Note that this list generation right here is why I ran a different
%indexing routine for Nodal than previous scripts.  Namely, when trying to
%determine which nodal a given triangle contains, you want membership
%tests to be based on integers and NOT (x,y) coordinates.  (x,y)
%coordinates have numerical round-off error that the computer can't
%distinguish between. But integers are unambiguous.  
count = size(Nodal,1);
Ntri = zeros(numtri,6); %Three for the vertices and three for the edge midpoints.

%Goes over the vertices.
for i=1:numtri
    for j=1:3
        k=1;%Cycle through interior vertices
        while k<= Nodal_Bud(1,1) && tri(i,j)~= Nodal(k,1)
            k=k+1; %If still cycling through int vertices and not hit it, keep going.
        end
        if k<= Nodal_Bud(1,1) %We found it.
        Ntri(i,j) = k; %Index it to the nodal list.
        else
            k=Nodal_Bud(2,1)+1; %Cyle through the boundary vertices.
            while k<=Nodal_Bud(3,1) && tri(i,j)~=Nodal(k,1)
                k=k+1;
            end
            Ntri(i,j) = k; %Because it will have to be listed here.
        end
    end
end

%Goes over the edges
%Goes over the vertices.
for i=1:numtri
    for j=1:3
        k=Nodal_Bud(1,1)+1; %Cyle through interior edges
        while k<= Nodal_Bud(2,1) && Etri(i,j)~= Nodal(k,1)
            k=k+1; %If still cycling through int vertices and not hit it, keep going.
        end
        if k<= Nodal_Bud(2,1) %We found it.
        Ntri(i,j+3) = k; %Index it to the nodal list.
        else
            k=Nodal_Bud(3,1)+1; %Cyle through the boundary vertices.
            while k<=Nodal_Bud(4,1) && Etri(i,j)~=Nodal(k,1)
                k=k+1;
            end
            Ntri(i,j+3) = k; %Because it will have to be listed here.
        end
    end
end

%%%%%%% Nmem %%%%%%%%%
%Here the goal is to write a script which indicates to which triangles a
%given node belongs as indexed on nodes

Nmem = zeros(size(Nodal,1),numtri); %Again max triangles can have membership for

for i=1:numtri
    %In the ith triangle, fill in the nodal indices in the space available
    for j=1:6
        k=1;
        while Nmem(Ntri(i,j),k) ~= 0 && k < numtri
            k = k + 1;
        end
    Nmem(Ntri(i,j),k) = i;
    end
end

%Cut out the unnecessary zeros
S = sum(Nmem);
k=1;
while k<= size(Nmem,2) && S(1,k)~=0
    k = k + 1;
end
Nmem(:,k:numtri) = [];
    

end



